'Programm ......... : Elektronische Sicherung ESS_Z
'
'Programminhalt ... : - Einlesen des stromproportionalen Spannungswertes eines
'                       Stromsensors
'                     - Funktionen:
'                       > Betriebsartenwahl (4 Tasten-Bedienung)
'                       > Ein- und Ausschalten
'                       > Stromberwachung
'                       > LCD-Steuerung (Hintergrundbeleuchtung und Kontrast)
'                     - frei konfigurierbare Kanle (analog, digital)
'
'Programmierer .... : Ingolf Bauer (ingolf.bauer@nexgo.de)
'Version .......... : 1.0.67
'Datum ............ : 28.02.2012
'
'-------------------------------------------------------------------------------
'
'Simulator-Einstellmglichkeit
'-----------------------------
'$sim                                              'Wartezeit aus
'
' Achtung!
' Diese Zeile unbedingt auskommentieren, bevor der AVR programmiert wird, da
' sonst der AVR nicht korrekt arbeitet.
' Die Aktivierung ist nur fr den Simulator vorgesehen.
'
'-------------------------------------------------------------------------------
'
'Versionsbersicht
'-----------------
'
'
'
'
'
'
'V 1.0.67  28.02.2012 - Information . : - Ausgabeversion
'V 1.0.99  18.02.2012 - Information . : - Beginn der Softwareentwicklung
'- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
'
'Programmspeicher des MCs (Programm-Flashspeicher): V 1.0.67 - 56%
'
'-------------------------------------------------------------------------------
'
'Informationen zur Software
'--------------------------
' Inhalt                                                                | Zeile
'-------------------------------------------------------------------------------
' 01. Allgemeine Informationen ........................................ |  129
'      Taktfrequenz ................................................... |  132
'      Programmierung ................................................. |  137
'      Analog-Digital-Umsetzer ........................................ |  145
' 02. Abkrzungsbersicht ............................................. |  159
' 03. Compilereinstellungen ........................................... |  170
' 04. Funktionsbeschreibung ........................................... |  195
'      Allgemeines .................................................... |  198
'      Anzeige und Bedienung .......................................... |  205
' 05. Interrupt ....................................................... |  256
' 06. Einstellung der FUSE- und/oder LOOK-Bits ........................ |  261
' 07. Anschluschema .................................................. |  276
'      AVR ATmega16 ................................................... |  279
'                        -------------------|_|------------------- .... |  281
' 08. I/O-Ports ....................................................... |  306
'      Abkrzungen fr die Signalbezeichnung .......................... |  309
'      Portbelegung ................................................... |  343
' 09. Steckverbinder .................................................. |  408
'      X1 Einspeisung ................................................. |  411
'      X2 Ausgang ..................................................... |  417
'      X4 LCD-Anschluss ............................................... |  423
'      X5 Programmierung .............................................. |  437
'      X7 Erweiterung ................................................. |  451
' 10. Definition ...................................................... |  466
'      Schaltkreis .................................................... |  469
'      Schnittstelle .................................................. |  474
'      Compilereinstellungen .......................................... |  479
' 11. Deklaration ..................................................... |  486
' 12. Konfiguration ................................................... |  568
'      I/O-Port ....................................................... |  571
'      Timer .......................................................... |  606
'      ADC ............................................................ |  624
'      LCD ............................................................ |  633
'      Interrupt ...................................................... |  644
' 13. Alias ........................................................... |  655
'      Eingang (digital) .............................................. |  658
'      Ausgang (digital) .............................................. |  663
' 14. Initialisierung ................................................. |  678
'      Blinken ........................................................ |  681
'      EEprom-Variablen auslesen ...................................... |  692
'      Ausgabe der Softwareinformation ................................ |  707
'      Aktivierung nach der Startroutine .............................. |  722
' 20. Programm ........................................................ |  742
'      Tastenabfrage .................................................. |  746
'      Abschaltung .................................................... |  755
'      Berechnung der Abschaltspannung (Stromberwachung) ............. |  783
'      Messung der Spannung und Anzeige ............................... |  803
'      Spannungswert am Strom-Sensor (HLS: Aus) ....................... |  818
'      Spannungswert am Strom-Sensor (HLS: Ein) ....................... |  827
'      Berechnung des Spannungswertes (Ausgang) ....................... |  836
'      Berechnung des Stromwertes (Ausgang) ........................... |  847
'      Anzeige der Messwerte .......................................... |  861
' 30. Unterprogramm ................................................... |  912
'      Taste .......................................................... |  916
'      Taste 1 - kurz (1110_xxxx): 224 ................................ |  937
'      Taste 2 - kurz (1101_xxxx): 208 ................................ |  954
'      Taste 3 - kurz (1011_xxxx): 176 ................................ |  975
'      Taste 4 - kurz (0111_xxxx): 112 ................................ |  997
'      Taste 1 - lang (1110_xxxx): 224 ................................ | 1012
'      Taste 2 - lang (1101_xxxx): 208 ................................ | 1018
'      Taste 3 - lang (1011_xxxx): 176 ................................ | 1068
'      Taste 4 - lang (0111_xxxx): 112 ................................ | 1082
' 40. Interruptroutine ................................................ | 1132
'      Timer 0 ........................................................ | 1136
'      PWM-Steuerung fr das LCD (Kontrast) ........................... | 1162
'      PWM-Steuerung fr das LCD (Helligkeit) ......................... | 1169
'      Prfung des Stromwertes ........................................ | 1179
'      Timer 1 ........................................................ | 1190
'      Zeitzhler fr die LCD-Beleuchtung ............................. | 1215
'      Zeitzhler fr die LCD-Kontraststeuerung ....................... | 1223
'      Blinken ........................................................ | 1228
' 50. Unterprogramm-Call .............................................. | 1274
'      Hintergrundbeleuchtung ......................................... | 1278
'      Tonausgabe ..................................................... | 1290
'      LCD-Werte normieren ............................................ | 1299
'      LCD-Anzeige .................................................... | 1310
' 60. Funktion ........................................................ | 1347
' 70. Datenbereich .................................................... | 1351
'-------------------------------------------------------------------------------
'
'01. Allgemeine Informationen
'----------------------------
'Taktfrequenz
'------------
' Als Taktfrequenz wir eine Quarzfrequenz von 16,0 MHz verwendet.
' Bei einer anderen Frequenz sind die Timer-Einstellunge anzupassen.
'
'Programmierung
'--------------
' Die Programmierung des MCs kann extern oder ber die vorhandene Schnittstelle
' erfolgen.
' Fr die Programmierung ist das Modul "USBasp" verwendbar, welches ber eine
' USB-Schnittstelle an einen PC angeschlossen wird. Es kann direkt ber eine,
' in BASCOM integrierte Einstellung angesprochen werden.
'
'Analog-Digital-Umsetzer
'-----------------------
'25.02.2012 - Quelle:
'             http://halvar.at/elektronik/kleiner_bascom_avr_kurs/adc_mit_getadc
'
'             Der Befehl GETADC stellt zuerst den Multiplexer ein. Danach wird
'             erst gemessen. Es knnte also passieren, dass ein Interrupt das
'             Programm genau zwischen dem Umstellen des Multiplexers und der
'             Messung unterbricht. Das alleine ist noch kein Problem. Es wird
'             nur dann zu einem Problem, wenn im Interrupt-Handler der Multi-
'             plexer auf einen anderen Pin umgeschaltet wird. Falls also der
'             Multiplexer innerhalb eines Interrupt-Handlers umgestellt wird
'             (warum auch immer), sollte vor dem Verwenden von GETADC die Inter-
'             rupts global aus- und danach wieder eingeschaltet werden.
'
'02. Abkrzungsbersicht
'-----------------------
' ADU ... : Analog-Digital-Umsetzer
' BA  ... : Betriebsart
' ESS_Z . : Elektronische Sicherung mit Zusatzfunktionen
' HGB ... : LCD-Hintergrundbeleuchtung
' HLS ... : Halbleiterschalter
' LCD ... : LC-Display
' MC .... : Mikrocontroller
' PWM ... : Pulsweitenmodulation

'03. Compilereinstellungen
'-------------------------
' Hinweise fr ATmega8/ATmega16 mit 1024 Byte SRAM
' Sie gelten ggf. auch fr den hier eingesetzten Mikrocontroller.
'
' Quelle:
' //halvar.at/elektronik/kleiner_bascom_avr_kurs/speicher_hwstack_swstack_frame/
'
' HW-Stack    - Empfehlung: 40 Byte bei INT, sonst 10 Byte ausreichend
'  32 Byte fr INT + je GOSUB 2 Byte, wenn bei INT ein GOSUB erfolgt
'  (bei Verschachtelung ... + n x 2 Byte)
'
' SW-Stack    - Empfehlung: 32 Byte (recht komplexe Aufrufe werden abgedeckt)
'   2 Byte fr jede bergebene Variable einer GOSUB-Routine + 2 Byte fr jede
'   LOCAL-Variable innerhalb einer GOSUB
'   (bei Verschachtelung ... n x 2 Byte)
'
' Frame-Space - Empfehlung: 60 Byte sollten alles abdecken
'                           (bei wenig Speicherverkleinern)
'  Beispiel:
'   2 x BYTE        =  2 Byte (erster und zweiter Parameter)
'   1 x WORD        =  2 Byte (dritte Parameter)
'   1 x STRING * 10 = 11 Byte (inkl. Abschlussbyte; vierter Parameter)
'   3 x BYTE        =  3 Byte (lokale Variablen)
'
'04. Funktionsbeschreibung
'-------------------------
'Allgemeines
'-----------
' Stromversorgung
' Fr die Spannungsstabilisierung wurde ein SEPIC-Wandler eingesetzt. Toleranz-
' bedingte Abweichungen der Bauelemente wirken sich auf die die Funktion des
' ESS_Z (z. B. Abschaltwerte) aus und sind ggf. im Programm anzupassen.
'
'Anzeige und Bedienung
'---------------------
' Wird die ESS_Z an eine Spannung angschlossen, startet das Programm im MC.
' Dies wird durch ein "Lebensbit" angezeigt, ein kurzes regelmiges Aufleuchten
' der vorhandenen LED. Ist ein LCD angeschlossen, wird die Softwareversion und
' die aktuelle Betriebsart angezeigt.
' In dieser Version werden ber die Taster folgende Funktionen aufgerufen:
'
' Taste | Bettigung kurz               HW | Bettigung lang               HW
'----------------------------------------------------------------------------
'  T1   |  Ein- bzw. Ausschalten           |  -
'  T2   |  Stromwerte erhhen           1) |  Wahl der Betriebsart         2)
'  T3   |  LCD Kontrasteinstellung      3) |  LCD Helligkeit der HGB       4)
'  T4   |  Einschalten der HGB          5) |  Besttigung                  6)
'
' Hinweis
' 1) In der BA 2 sind Schritte von 50 mA, in der BA 3 Schritte von 250 mA ein-
'    gestellt. Die Begrenzung liegt bei 5.000 mA und kann bei Einsatz eines an-
'    deren Sensors angepasst werden.
' 2) Die Nummer der BA wird bei 1 gestartet und automatisch bis 20 erhht.
'    Je nach BA erfolgt der automatische Aufruf oder muss mit der Taste 4
'    besttigt werden.
' 3) Beim ersten Drcken wird die Richtung der Kontrasteinstellung verndert.
'    Weitere kurze Tastenbedienungen ndern den Wert. Wird eine kurze Pause ge-
'    macht, kehrt sich die Richtung der Kontrasteinstellung wieder um. Auf diese
'    Art und Weise kann der Kontrast von "gering" zu "stark" (und umgekehrt)
'    angepasst werden.
' 4) Wie bei der Kontrasteinstellung erfolgt auch die Einstellung der LCD-Hin-
'    tergrundbeleuchtung durch die Vernderung der PWM-Werte.
' 5) Bei dem festgelegten Wert wird die HGB fr 20 s eingeschaltet. Auerdem
'    wird bei jedem anderen Tastendruck die HGB fr eine krzer Zeit eingeschal-
'    tet. Eine bereits laufende Zeit wird nicht verringert.
' 6) Dieser Taste ist je nach eingestellter BA eine "Besttigungsfunktion" zuge-
'    wiesen.
'
' Da bei dem eingesetzten Halbleiterschalter ein Strom unterhalb von 300 mA so
' intepretiert wird, als wenn kein Verbrauche angeschlossen wre, erfolgt bis zu
' 500 mA keine berwachung des Status-Signals. Zu beachten ist in diesem Fall,
' dass auch andere Fehler vom Programm nicht erkannt werden (siehe bitte Daten-
' blatt des BTS).
'
' Der eingeschaltete HLS wird mit einem gelben Dauerlicht, der gestrte durch
' ein rotes Blinklicht angezeigt. Nach der Fehlerbeseitigung (z. B. Erhhung
' des Lastwiderstandes) kann mit Taste 1 der ESS_Z wieder eingeschaltet werden.
'
' Anmerkung
' Der folgende Programmcode soll als Anregungen fr eigene Softwarelsungen
' dienen.
' Sind umfangreichere Funktionen vorgesehen, kann es sein, dass der Flash-Spei-
' nicht mehr ausreicht. In diesem fall kann der pinkompatible ATmega32 einge-
' setzt werden.
'
'05. Interrupt
'-------------
' Timer 0:  8 Bit, Interrupt bei berlauf (interne Takterzeugung: 1 ms)
' Timer 1: 16 Bit, Interrupt bei berlauf (Zeitbasis)
'
'06. Einstellung der FUSE- und/oder LOOK-Bits
'--------------------------------------------
' Fr die Einstellung der Fuse- und Lookbits ist der Bauanleitung ein "screen-
' shot" (Programmer: USBasp) beigelegt.
'
' Anpassungen sind bei der Verwendung des USBasp mit einem internen Brennpro-
' gramm von BASCOM mglich (z. B. Lschung des EEproms zusammen mit dem Brennen
' des Programms).
'
' /// Achtung ///
' Beim ATmega16 ist das JTAG(EN) Flag defaultmig auf Enable gesetzt. Dies
' bedeutet, dass das JTAG-Interface (prozenahe Datenabfrage) aktiviert ist.
' Dann kann aber Port C fr keine anderen Aufgaben genutzt werden!
' Daher muss bei Anschluss eines LCDs die Einstellung verndert werden.
'
'07. Anschluschema
'------------------
'AVR ATmega16
'------------
'                  -------------------|_|-------------------
' [0] LED_gn <--- | 1 [PB0]                         [PA0] 40| <--- SPW     [A]
' [0] LED_rt <--- | 2 [PB1]                         [PA1] 39| <--- STW     [A]
' [0]   PIZ  <--- | 3 [PB2, INT2]                   [PA2] 38| <--- STA     [1]
'       AS_1 <--> | 4 [PB3, OC0]                    [PA3] 37| ---> EIN     [0]
'       AS_2 <--> | 5 [PB4]                         [PA4] 36| <--> EW_5
' [~]   MOSI ---> | 6 [PB5, MOSI]                   [PA5] 35| <--> EW_6
' [~]   MISO <--- | 7 [PB6, MISO]                   [PA6] 34| <--> EW_7
' [~]   SCK  ---> | 8 [PB7, SCK]                    [PA7] 33| <--> EW_8
' [~] /RESET ---> | 9 [/Reset]                     [AREF] 32| ==== KREF
'      P5V_D ==== |10 [VCC]                        [AGND] 31| ==== GND
'        GND ==== |11 [GND]                        [AVCC] 30| ==== P5V_A
'         QZ ++++ |12 [XTAL2]                       [PC7] 29| ---> LCD_PWM
'         QZ ++++ |13 [XTAL1]                       [PC6] 28| ---> LCD_HGB
'       EW_1 <--> |14 [PD0, RXD]                    [PC5] 27| ---> LCD_DB7
'       EW_2 <--> |15 [PD1, TXD]                    [PC4] 26| ---> LCD_DB6
'       EW_3 <--> |16 [PD2, INT0]                   [PC3] 25| ---> LCD_DB5
'       EW_4 <--> |17 [PD3, INT1]                   [PC2] 24| ---> LCD_DB4
' [1]    T_1 ---> |18 [PD4]                         [PC1] 23| ---> LCD_RS
' [1]    T_2 ---> |19 [PD5]                         [PC0] 22| ---> LCD_E
' [1]    T_3 ---> |20 [PD6]                         [PD7] 21| <--- T_4     [1]
'                  -----------------------------------------
'
' Werte in []: Einstellungen nach dem Programmstart
'  ~: Impuls   0: aus   1: ein   A: analog
'
'08. I/O-Ports
'-------------
'Abkrzungen fr die Signalbezeichnung
'-------------------------------------
' /Reset  : Resetfunktion
' AS_     : Anschluss (frei konfigurierbar)
' EIN     : Steuerung des Halbleiterschalters
' EW_     : Erweiterung (Wannensteckverbinder)
' KREF    : Referenzspannung    (Kondensator)
' LCD_    : LC-Display
' LED_    : Anschluss LED
' PIZ     : Anschluss Piezo-Signalgeber
' SPW     : Spannungswert       (analoger Kanal)
' STW     : Stromwert           (analoger Kanal)
' STA     : Status              (Pegel bei Ausfall: 0)
' T_      : Taster
'
' -<n>    : diverse (n: 1-8)
' _A      : analog
' _D      : digital
' _DBn    : LCD-Datenbit
' _E      : LCD-Steuersignal
' _gn     : grn
' _HGB    : LCD-Hintergrundbeleuchtung
' _PWM    : LCD-Pulsweitenmodulation (Kontrast)
' _RS     : LCD-Steuersignal
' _rt     : rot
'
' P5V_    : Stromversorgung P 5V
' GND     : Stromversorgung Masse
'
' MISO    : Programmieranschlu
' MOSI    : Programmieranschlu
' QZ      : Anschlu Quarz
' SCK     : Programmieranschlu
'
'Portbelegung
'------------
'           Funktion     | Signal        | Erluterung                    | Pin
' - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
' -   : SV  P5V_A ...... | ............. | Stromversorgung analog ....... | 30
' -   : SV  P5V_D ...... | ............. | Stromversorgung digital ...... | 10
' -   : SV  GND ........ | ............. | Stromversorgung .............. | 11
' -   : SV  GND ........ | ............. | Stromversorgung .............. | 31
' -   : PRG /RESET ..... | ............. | Programmierung/Reset des MCs . |  9
' -   : SV  Kondensator  | KREF ........ | Referenzspannung fr ADU ..... | 32
' -   : TK  Takt ....... | QZ .......... | Quarzanschluss 1 ............. | 12
' -   : TK  Takt ....... | QZ .......... | Quarzanschluss 2 ............. | 13
'
'Port A     Funktion     | Signal        | Erluterung                    | Pin
' - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
' - .0: AE  Spannung ... | SPW ......... | Ausgangsspannung messen ...... | 40
' - .1: AE  Strom ...... | STW ......... | Ausgnagsstrom messen ......... | 39
' - .2: DE  Status ..... | STA ......... | Status des HLS ............... | 38
' - .3: DA  Ansteuerung  | EIN ......... | Ansteuerung des HLS .......... | 37
' - .4: Kx  (ohne) ..... | EW_5 ........ | .............................. | 36
' - .5: Kx  (ohne) ..... | EW_6 ........ | .............................. | 35
' - .6: Kx  (ohne) ..... | EW_7 ........ | .............................. | 34
' - .7: Kx  (ohne) ..... | EW_8 ........ | .............................. | 33
'
'Port B     Funktion     | Signal        | Erluterung                    | Pin
' - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
' - .0: DA  Anzeige gn . | LED_gn ...... | Anzeige von Zustnden ........ |  1
' - .1: DA  Anzeige rt . | LED_rt ...... | Anzeige von Zustnden ........ |  2
' - .2: DA  Tonausgabe . | PIZ ......... | Signalton .................... |  3
' - .3: Kx  (ohne) ..... | AS_1 ........ | .............................. |  4
' - .4: Kx  (ohne) ..... | AS_2 ........ | .............................. |  5
' - .5: PRG MOSI ....... | MOSI ........ | Programmierung ............... |  6
' - .6: PRG MISO ....... | MISO ........ | Programmierung ............... |  7
' - .7: PRG SCK ........ | SCK ......... | Programmierung ............... |  8
'
'Port C     Funktion     | Signal        | Erluterung                    | Pin
' - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
' - .0: DA  LCD ........ | LCD_E ....... | LCD-Steuersignal Enable ...... | 22
' - .1: DA  LCD ........ | LCD_RS ...... | LCD-Steuersignal RS .......... | 23
' - .2: DA  LCD ........ | LCD_DB4 ..... | LCD-Datenbit 4 ............... | 24
' - .3: DA  LCD ........ | LCD_DB5 ..... | LCD-Datenbit 5 ............... | 25
' - .4: DA  LCD ........ | LCD_DB6 ..... | LCD-Datenbit 6 ............... | 26
' - .5: DA  LCD ........ | LCD_DB7 ..... | LCD-Datenbit 7 ............... | 27
' - .6: DA  LCD ........ | LCD_HGB ..... | LCD-Hintergrundbeleuchtung ... | 28
' - .7: DA  LCD ........ | LCD_PWM ..... | LCD-Kontrasteinstellung ...... | 29
'
'Port D     Funktion     | Signal        | Erluterung                    | Pin
' - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
' - .0: Kx  (ohne) ..... | EW_1 ........ | .............................. | 14
' - .1: Kx  (ohne) ..... | EW_2 ........ | .............................. | 15
' - .2: Kx  (ohne) ..... | EW_3 ........ | .............................. | 16
' - .3: Kx  (ohne) ..... | EW_4 ........ | .............................. | 17
' - .4: DE  Taster ....  | T_1 ......... | Befehlseingabe 1 ............. | 18
' - .5: DE  Taster ....  | T_2 ......... | Befehlseingabe 2 ............. | 19
' - .6: DE  Taster ....  | T_3 ......... | Befehlseingabe 3 ............. | 20
' - .7: DE  Taster ....  | T_4 ......... | Befehlseingabe 4 ............. | 21

'Erluterung
' B  : Bussystem
' DA : digitaler Ausgang
' DE : digitaler Eingang
' Kx : Kanal ist frei konfigurierbar
' PRG: Programmierung
' QZ : Quarzanschluss
' SV : Stromversorgung
'
'09. Steckverbinder
'------------------
'X1 Einspeisung
'--------------
' Pin   Belegung
'   1    Versorgungsspannung (+Ub)
'   2    Versorgungsspannung (Masse)
'
'X2 Ausgang
'----------
' Pin   Belegung
'   1    Ausgang (+Ub)
'   2    Ausgang (Masse)
'
'X4 LCD-Anschluss
'----------------
' Pin   Belegung
'   1    LCD_DB4
'   2    +Ub
'   3    LCD_DB5
'   4    LCD_RS
'   5    LCD_DB6
'   6    LCD_E
'   7    LCD_DB7
'   8    LCD_KON
'   9    LCD_HGB
'  10    Masse
'
'X5 Programmierung
'-----------------
' Pin   Belegung
'   1    PRG_MOSI
'   2    - (ggf. mit +Ub per Brcke auf der Leiterplatte verbinden)
'   3    -
'   4    -
'   5    PRG_/Reset
'   6    LCD_E
'   7    PRG_SCK
'   8    Masse
'   9    PRG_MISO
'  10    Masse
'
'X7 Erweiterung
'--------------
' Pin   Belegung
'   1    EW_01 (AVR PD0, RXD)
'   2    EW_02 (AVR PD1, TXD)
'   3    EW_03 (AVR PD2, INT0)
'   4    EW_04 (AVR PD3, INT1)
'   5    EW_05 (AVR PA4, ADC4)
'   6    EW_06 (AVR PA5, ADC5)
'   7    EW_07 (AVR PA6, ADC6)
'   8    EW_08 (AVR PA7, ADC7)
'   9    +Ub
'  10    Masse
'
'-------------------------------------------------------------------------------
'
'10. Definition
'--------------
'Schaltkreis
'-----------
   $regfile = "m16adef.dat"                       'Prozessor ....... : ATmega16
   $crystal = 16000000                            'Frequenz ........ : 16,0 MHz

'Schnittstelle
'-------------
' ggf. fr eine Ausgabe im Simulator-Feld (UART)
' $baud = 9600                                   'Baudrate ........ : 9600
                                                 '9600/19200/38400/57600/115200
'Compilereinstellungen
'---------------------
   $hwstack = 40                                  'Hardware stack
   $swstack = 32                                  'Software stack
   $framesize = 60                                'Frame space
'
'-------------------------------------------------------------------------------
'
'11. Deklaration
'---------------
'Subrotine (Call)
' LCD-Beleuchtung
   Declare Sub Lcd_b(byval T_hgb As Byte)
' Tonausgabe (Piep)
   Declare Sub Ton_d(byval T_ton As Word )
' LCD-Werte normieren
   Declare Sub Wanz(byval Wsin As Single)
' LCD-Anzeige
   Declare Sub Lcd_t(byval Tul As String , Byval Tll As String)

'Flag
   Dim F_amw As Bit                               'Anzeige Messwert
   Dim F_ast As Bit                               'Anzeige Strung
   Dim F_ba As Bit                                'Betriebsart

   Dim F_hel As Bit                               'Richtung Helligkeit
   Dim F_hgb As Bit                               'LC-Beleuchtung
   Dim F_kon As Bit                               'Richtung Kontrast

   Dim F_lb As Bit                                'Lebensbit
   Dim F_taste As Bit                             'Taste
   Dim F_ea As Bit                                'Ein/Aus

   Dim F_feh As Bit                               'Fehler
   Dim F_sta As Bit                               'Status
   Dim F_adc As Bit                               'ADC-Abfrage

'Konstante
   Dim K_bts As Single                            'Datenblatt: 186 mV/A

'Variable
   Dim Anz_i As Single                            'Strom-Anzeige
   Dim Anz_p As Single                            'Lesitungs-Anzeige
   Dim Anz_r As Single                            'Widerstand-Anzeige

   Dim Anz_u As Single                            'Spannung-Anzeige
   Dim Ba As Byte                                 'Betriebsart
   Dim Imax As Word                               'maximaler Stromwert

   Dim Imax_old As Word                           'alter Stromwert
   Dim Pwm_hgb As Byte                            'LCD: Helligkeit
   Dim Pwm_kon As Byte                            'LCD: Kontrastwert

   Dim Taste As Byte                              'Taste
   Dim Taste_old As Byte                          'Taste (Merker)
   Dim Uadc As Word                               'Spannungsmessung

   Dim Uimes As Word                              'Strom-Messwert
   Dim Uss As Word                                'Stromsensor-Spannung
   Dim Ui As Single                               'Spannung fr Stromabschaltung

   Dim Uiab As Word                               'Vergleichswert

'Text
   Dim Hs01 As String * 5                         'HS01 fr Anzeige
   Dim Hs02 As String * 5                         'HS02 fr Anzeige
   Dim Hs01_lcd As String * 16                    'HS fr LCD

   Dim Hs02_lcd As String * 16                    'HS fr LCD
   Dim Tul_1 As String * 16                       'Text-upperline
   Dim Tll_1 As String * 16                       'Text-lowerline

   Dim Tul_old As String * 16                     'Text-upperline (bisher)
   Dim Tll_old As String * 16                     'Text-lowerline (bisher)

'Zhler
   Dim Hz01 As Byte                               'Hilfszhler 01
   Dim Z_bl As Byte                               'Zeit T1 (Blinken)
   Dim Z_hgb As Byte                              'LCD-Beleuchtung

   Dim Z_kon As Byte                              'Wartezeit Richtung
   Dim Z_t0a As Byte                              'Zeit T0a
   Dim Z_t0b As Byte                              'Zeit T0b

   Dim Z_t1 As Byte                               'Zeit T1
   Dim Z_t2 As Byte                               'Zeit T2
   Dim Z_taste As Byte                            'Lnge Tastendruck
'
'-------------------------------------------------------------------------------
'
'12. Konfiguration
'-----------------
'I/O-Port
'--------
'=Port A
'-Datenrichtung (Ein [0]: Eingang, Aus [1]: Ausgang)
   Ddra = &B0000_1000                             'PA0-2,4-7 Ein; PA3 Aus
'-Pullup-Widerstand (passiv [0], aktiv [1])
   Porta = &B0000_0100                            'PA0-1,3-7 passiv, PA2 aktiv
'  Widerstnde an
'   PA1: deaktivieren (sonst zu hohe Messspannung)
'   PA2: aktivieren fr korrektes Einlesen des Pegels
'- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
'=Port B
'-Datenrichtung (Ein [0]: Eingang, Aus [1]: Ausgang)
   Ddrb = &B0000_0111                             'PB3-7 Ein; PB0-2 Aus
'-Pullup-Widerstand (passiv [0], aktiv [1])
   Portb = &B0000_0000                            'PB0-7 passiv
'- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
'=Port C
'-Datenrichtung (Ein [0]: Eingang, Aus [1]: Ausgang)
   Ddrc = &B1111_1111                             'PC0-7 Aus
'-Pullup-Widerstand (passiv [0], aktiv [1])
   Portc = &B0011_1111                            'PC0-5 aktiv; PC6-7 passiv
'//Achtung//
'  nicht fr folgende Ausgnge, da PWM-moduliert:
'   PC6 : Kontrast (positive Spannung => Kontrast reduziert)
'   PC7 : LCD-Hintergrundbeleuchtung (HGB)
'- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
'=Port D
'-Datenrichtung (Ein [0]: Eingang, Aus [1]: Ausgang)
   Ddrd = &B0000_0000                             'PD0-7 Ein
'-Pullup-Widerstand (passiv [0], aktiv [1])
   Portd = &B1111_0000                            'PD0-3 passiv; PD4-7 aktiv
'
'-------------------------------------------------------------------------------
'
'Timer
'-----
'Normal-Konfiguration
'Die Timerinterrupts 0/1 werden beim jeweiligen Timerberlauf ausgelst.
'                                  PSC  TVE   T         f
'T0 = 1/ Quarz* Teiler (1/ 16 MHz* 256* 193 =  1,00 ms; 992 Hz)
'T1 = 1/ Quarz* Teiler (1/ 16 MHz* 256* 625 = 10,00 ms; 100 Hz)

'Timer0 voreinstellen auf Wert: 256-TVE
   Config Timer0 = Timer , Prescale = 256         'Konfiguration
      Timer0 = 193                                'Voreinstellung

'Timer1 voreinstellen auf Wert: 65536-TVE
   Config Timer1 = Timer , Prescale = 256         'Konfiguration
      Timer1 = 64911                              'Voreinstellung
'
'-------------------------------------------------------------------------------
'
'ADC
'---
'Auto ............. : Einstellung des hchstmglichen Taktes fr den ADC
'Referenzspannung . : Reference= Internal . : 2,56 V
'                                Avcc ..... : Vcc
   Config Adc = Single , Prescaler = Auto , Reference = Internal
'
'-------------------------------------------------------------------------------
'
'LCD
'---
'LCD-Typ .......... : 2 Zeilen, 16 Zeichen
   Config Lcd = 16 * 2
   Config Lcdpin = Pin , Db4 = Portc.2 , Db5 = Portc.3 , Db6 = Portc.4 , _
          Db7 = Portc.5 , E = Portc.0 , Rs = Portc.1
   Cursor Off
   Cls                                            'Anzeige lschen
'
'-------------------------------------------------------------------------------
'
'Interrupt
'---------
   Enable Timer0                                  'Timer 0: Interrupt
      On Timer0 Int_tim0                          'Timer 0: berlauf

   Enable Timer1                                  'Timer 1: Interrupt
      On Timer1 Int_tim1                          'Timer 1: berlauf

   Enable Interrupts                              'Interrupts: global zulassen
'
'-------------------------------------------------------------------------------
'
'13. Alias
'---------
'Eingang (digital)
'-----------------
'Port A
   Bts_sta Alias Pina.2                           'Status HLS (Pegel 0: Fehler)

'Ausgang (digital)
'-----------------
'Port A
   Hls Alias Porta.3                              'Halbleiterschalter

'Port B
   Led_gn Alias Portb.0                           'LED-Anzeige grn
   Led_rt Alias Portb.1                           'LED-Anzeige rot
   Ton Alias Portb.2                              'Piezo-Signalgeber

'Port C
   Lcd_hgb Alias Portc.6                          'LCD Hintergrundbeleuchtung
   Lcd_kon Alias Portc.7                          'LCD Kontrast
'
'-------------------------------------------------------------------------------
'
'14. Initialisierung
'-------------------
'Blinken
'-------
'beim Programmstart bzw. nach einem Reset
'Lebensbit
   F_lb = 0                                       'bei Initialisierung inaktiv

   For Hz01 = 1 To 6
      Led_gn = Not Led_gn                         'LED_gn (3x)
      Waitms 500
   Next Hz01

'EEprom-Variablen auslesen
'-------------------------
'-Kontrast des LCD
   Readeeprom Pwm_kon , &H01                      'Byte (1 Byte)
      If Pwm_kon > 20 Then Pwm_kon = 20           'Erststart: Wert 20
'-Helligkeit des LCD (Hintergrundbeleuchtung)
   Readeeprom Pwm_hgb , &H02                      'Byte (1 Byte)
      If Pwm_hgb > 20 Then Pwm_hgb = 20           'Erststart: Wert 20
'-Betriebsart
   Readeeprom Ba , &H03                           'Byte (1 Byte)
      If Ba > 20 Then Ba = 1                      'Erststart: Wert 1
'-Abschaltwert Strom
   Readeeprom Imax , &H04                         'Word (2 Byte)
      If Imax > 5000 Then Imax = 5000             'Erststart: Wert 5000

'Ausgabe der Softwareinformation
'-------------------------------
   Call Lcd_b(200)                                'LCD-Beleuchtung: 20 s

   Call Lcd_t( "Amatronik - ESSz" , "Version: 1.0.67")
   Wait 2

   Hs01_lcd = Str(ba)
   Hs01_lcd = Trim(hs01_lcd)
   Hs01_lcd = "Betriebsart: " + Hs01_lcd
   Call Lcd_t(hs01_lcd , "Wahl mit T2_lang")

   Call Ton_d(100)                                'Ton (100 ms)
   Wait 3                                         'Anzeigezeit

'Aktivierung nach der Startroutine
'---------------------------------
   F_lb = 1                                       'Lebensbit
   F_kon = 1                                      'Kontrast (Richtung)
   F_sta = 1                                      'Status
   Uimes = 1023                                   'Startwert
   Uss = 1023                                     'Startwert
   K_bts = 0.186                                  'BTS [V/A], Datenblatt: 0.186

'ggf. Umstellung der Betriebsart auf Normalbetrieb
   Select Case Ba
      Case 2 To 3
         Ba = 1
      Case Is > 10
         Ba = 1
   End Select

   Writeeeprom Ba , &H03                          'Wert im EEprom speichern
'
'-------------------------------------------------------------------------------
'
'20. Programm
'------------
Do
'Tastenabfrage
'-------------
'Der Aufruf wird ber einen Timer-Interrupt gesteuert.
M_ka:
   If F_taste = 1 Then                            'Flag abfragen
      Gosub Up_taste
      F_taste = 0                                 'Reset Flag
   End If

'Abschaltung
'-----------
'Strung HLS oder Stromwert berschritten
'Das Flag F_feh wird bei einer Stromwertberschreitung (INT-Routine) gesetzt.
   If F_feh = 1 Or F_sta = 0 Then                 'Abfrage: Flag Strung, Status
      Hls = 0                                     'HSL Aus
      F_ea = 0                                    'Flag_EinAus Aus
      F_lb = 0                                    'Flag_Lebensbit Aus

      If Z_bl = 0 Then
         If F_feh = 1 Then
            Call Lcd_t( "Stromabschaltung" , "")
         End If

         If F_sta = 0 Then
            Call Lcd_t( "Statusfehler" , "(Schalter)")
         End If
      End If
   End If

'keine Anzeige, wenn eine Taste bettigt wird
   Hz01 = Pind And &B1111_0000                    'Kanal D: 4-7 abfragen
   If Hz01 <> 240 Then                            '240: keine Taste bettigt
      Goto M_ka                                   'zur Tastenabfrage
   Else
      F_ba = 0                                    'BA Einstellung
   End If

'Berechnung der Abschaltspannung (Stromberwachung)
'--------------------------------------------------
'Abschaltung des HLS, wenn der Grenzwert unterschritten wurde
'Wahl der Referenzspannung (Genauigkeit bei 2^10 = 1024 Stufen):
' 2,56 V       - ergibt: 2,56 V/ 1024 = 2,5000*EXP-3 V/Stufe)
' 5,00 V (Vcc) - ergibt: 5,00 V/ 1024 = 4,8828*EXP-3 V/Stufe)

'Die Dimensionierung gilt fr einen 5 A Sensor.
'Entsprechend der Einbaulage und der Angabe im Datenblatt verringert sich die
'Ausgangsspannung des Stromsensors von 2,45 V um 0,186 V/A (Ua=2,45V bei Ia=0A).
'Bei Imax= 5.000 mA ergibt die Berechnung einen Wert von 608, bei dem der HLS
'deaktiviert werden muss.
   If Imax_old <> Imax Then
      Imax_old = Imax
      Ui = Imax * K_bts                           'Datenblatt: 186 mV/A
      Ui = 2450 - Ui                              'Vcc/2
      Ui = Ui / 2.50                              '2,5 mV je Stufe
      Uiab = Int(ui)                              'neuer Wert aktiv (Interrupt)
   End If

'Messung der Spannung und Anzeige
'--------------------------------
'Die Spannung am Ausgang des HLS wird ermittelt. Dazu wird die am Kanal PA0
'die anliegende Spannung ermittelt.
'Als Vergleichspannung dient die interne Referenzspannung von 2,56 V; vorge-
'schaltet ist ein Spannungsteiler (16 k/ 1 k).
'Die Spannungsermittlung erfolgt alle 100 ms. In 3 Schritten zu je 10 ms wird
'der Mittelwert gebildet.
' Mittelwert: /2
' 2,5 mV/bit: /400 = 1/0.0025 V
' 16 k/ 1 K : *17
'
'Der Strom wird innerhalb einer Interruptroutine verglichen, um verzgerungsarm
'den HLS deaktivieren zu knnen.

'Spannungswert am Strom-Sensor (HLS: Aus)
'----------------------------------------
   If Hls = 0 Then                                'kein Stromfluss
      Uadc = Getadc(1)                            'Spannungswert
      Uss = 9 * Uss
      Uss = Uss + Uadc
      Uss = Uss / 10                              'Mittelwert mit Wichtung
   Else

'Spannungswert am Strom-Sensor (HLS: Ein)
'----------------------------------------
'Mittelwertbildung
      Uadc = Getadc(1)                            'Spannungswert
      Uimes = 9 * Uimes
      Uimes = Uimes + Uadc
      Uimes = Uimes / 10
   End If

'Berechnung des Spannungswertes (Ausgang)
'----------------------------------------
   If Ba = 1 Or Ba = 4 Then
   If F_amw = 1 And F_feh = 0 And F_sta = 1 Then
      Uadc = Getadc(0)                            'Spannungswert
      Uadc = Uadc + Getadc(0)
      Anz_u = Uadc
      Anz_u = Anz_u * 0.2125                      '/2, *17, *100, *0,0025 mV
      Anz_u = Int(anz_u)                          'ganzzahlig 10 mV
      Anz_u = Anz_u / 10                          'Anzeige Normieren

'Berechnung des Stromwertes (Ausgang)
'------------------------------------
      Anz_i = Anz_i * K_bts                       'Strom > Spannung
      Anz_i = Anz_i / 2.5                         'Spannung in ADC-Wert
      Anz_i = 9 * Anz_i                           'Wertigkeit
      Anz_i = Anz_i + Uss                         'Messwerte einrechnen
      Anz_i = Anz_i - Uimes
      Anz_i = Anz_i / 10

      Anz_i = Anz_i * 2.5                         'ADC mit 2,5 mV/Schritt
      Anz_i = Anz_i / K_bts                       'Datenblatt: 186 mV/A
      Anz_i = Int(anz_i)                          'ganzzahlig 10 mA
      Anz_i = Anz_i / 100                         'Anzeige normieren

'Anzeige der Messwerte
'---------------------
'Die Messwerte werden angezeigt, wenn die Betriebsart 1/4 gewhlt ist.
'Die Messwerte werden auf Null gesetzt, wenn der HLS nicht angesteuert wird.
      Select Case Ba
'Anzeige von Spannung und Strom
         Case 1
            If F_ea = 1 Then                      'HLS eingeschaltet
               Call Wanz(anz_u)
               Hs01_lcd = "Ua: " + Hs01 + " V"
               Call Wanz(anz_i)
               Hs02_lcd = "Ia: " + Hs01 + " A"
               Call Lcd_t(hs01_lcd , Hs02_lcd)
            Else
               Hs01_lcd = "Betriebsart " + Str(ba)
               Call Lcd_t( "Schalter aus" , Hs01_lcd)
            End If
'Anzeige von Widerstand und Leistung
         Case 4
            If Anz_i > 0 Then
'Widerstand
               Anz_r = Anz_u / Anz_i
               Anz_r = Int(anz_r)                 'Formatierung (x.00)
'Leistung
               Anz_p = Anz_u * Anz_i
               Anz_p = Anz_p * 10
               Anz_p = Int(anz_p)                 'Formatierung (x.x0)
               Anz_p = Anz_p / 10
            Else
               Anz_r = 0
               Anz_p = 0
            End If

            Call Wanz(anz_r)
               Hs01_lcd = "R: " + Hs01 + " Ohm"
            Call Wanz(anz_p)
               Hs02_lcd = "P: " + Hs01 + " Watt"
               Call Lcd_t(hs01_lcd , Hs02_lcd)
         End Select

      F_amw = 0                                   'Flag rcksetzen
   End If
   End If

Loop

'Ende des Programms
End

'===============================================================================

'-------------------------------------------------------------------------------
'30. === U N T E R P R O G R A M M =============================================
'-------------------------------------------------------------------------------

'Taste
'-----
Up_taste:
' Prfung, ob ein kurzer Druck oder die Taste lnger gehalten wird
      Taste_old = Taste                           'gedrckte Taste merken
      Taste = Pind And &B1111_0000                'Kanal D: 4-7 abfragen

   If Taste = 224 Or Taste = 208 Or Taste = 176 Or Taste = 112 Then
      If Z_taste < 100 Then
         Incr Z_taste                             'Zeitzhler
      End If                                      'bei einer gedrckten Taste
   End If

'Taste wurde nach einer kurzen Bettigung losgelassen
'++++++++++++++++++++++++++++++++++++++++++++++++++++
   If Taste <> Taste_old And Z_taste > 2 And Z_taste <= 20 Then
      Call Lcd_b(50)                              'LCD-Beleuchtung: 5 s
      Call Ton_d(10)                              'Tastenpiep

      Select Case Taste_old                       'gedrckte Taste

'Taste 1 - kurz (1110_xxxx): 224
'-------------------------------
'HSL schalten
         Case 224
'Das Schalten kann in jeder Betriebsart erfolgen.
            F_ea = Not F_ea                       'Umschalten des HSL
            F_feh = 0                             'ggf. Fehler lschen
'HLS und LED-Anzeige schalten
            If F_ea = 1 Then                      'kein Fehler, Flag HSL Ein
               Hls = 1                            'HSL Ein
            Else
               Hls = 0                            'HSL Aus
            End If

            F_lb = Not Hls                        'Flag Lebensbit
            Z_bl = 255                            'Zhler-Blinken rcksetzen

'Taste 2 - kurz (1101_xxxx): 208
'-------------------------------
'Entsprechend der gewhlten Betriebsart wird die Routine aufgerufen
         Case 208
            Select Case Ba
'2/3: Erhhung des Strom-Abschaltwertes um jeweils 50/250 mA
               Case 2 To 3
                  If Imax < 5000 Then
                     If Ba = 2 Then               'Betriebsart: 2
                        Imax = Imax + 50          ' 50 mA Schritt
                     Else                         'Betriebsart: 3
                        Imax = Imax + 250         '250 mA Schritt
                     End If
                  Else
                     Imax = 100
                  End If
                  If Imax > 5000 Then Imax = 5000
                  Hs01_lcd = "Imax: " + Str(imax) + " mA"
                  Call Lcd_t( "-" , Hs01_lcd)
            End Select

'Taste 3 - kurz (1011_xxxx): 176
'-------------------------------
'Kontrast: keine Richtungsumkehr, wenn innerhalb einer Zeit t eine Tastenbet-
'tigung stattgefunden hat.
         Case 176
            If Z_kon = 0 Then
               F_kon = Not F_kon
            End If

            Z_kon = 20                            'Wartezeit

'Fr die Kontrasteinstellung sind maximal 20 Stufen festgelegt.
'Mglich wren 50, aber dies ndert nichts mehr am Kontrast.
            If F_kon = 0 And Pwm_kon < 20 Then
               Incr Pwm_kon
            End If
            If F_kon = 1 And Pwm_kon > 0 Then
               Decr Pwm_kon
            End If

            Writeeeprom Pwm_kon , &H01            'Wert im EEprom speichern

'Taste 4 - kurz (0111_xxxx): 112
'-------------------------------
'Beleuchtung: einschalten (Zeitberwachung schaltet nach 20 s aus)
         Case 112
            Call Lcd_b(200)                       'LCD-Beleuchtung: 20 s
      End Select
   End If

'Taste gedrckt bei bereits lngere Bettigungszeit
'++++++++++++++++++++++++++++++++++++++++++++++++++
   If Taste = Taste_old And Z_taste > 20 Then
      Call Lcd_b(50)                              'LCD-Beleuchtung: 5 s
      Waitms 500
      Select Case Taste_old                       'gedrckte Taste

'Taste 1 - lang (1110_xxxx): 224
'-------------------------------
'+ + + Funktion noch frei definierbar + + +
         Case 224


'Taste 2 - lang (1101_xxxx): 208
'-------------------------------
'Die Betriebsarten werden nacheinander angezeigt.
         Case 208
            If Ba < 20 Then
               If F_ba = 0 Then
                  F_ba = 1
                  Ba = 0                          'Initialisierung: 0
               End If

               Incr Ba

               Hs01_lcd = "Betriebsart: " + Str(ba)
               Call Lcd_t(hs01_lcd , "")

               Select Case Ba
'1: Spannung und Strom
                  Case 1
                     Hs01_lcd = "Spannung / Strom"
'2/3: Strom-Abschaltwert (50/250 mA Schritte)
                  Case 2
                     Hs01_lcd = "I(50): " + Str(imax) + " mA"
                  Case 3
                     Hs01_lcd = "I(250): " + Str(imax) + " mA"
'4: Widerstand und Leistung
                  Case 4
                     Hs01_lcd = "Widerst./Leistg."
                  Case 19
                     Hs01_lcd = "Test LCD"
                  Case 20
                     Hs01_lcd = "Test LED/Piezo"
                  Case Else
                     Hs01_lcd = "(keine Funktion)"
                     Ba = 1
               End Select
               Call Lcd_t( "-" , Hs01_lcd)
            Else
'21:
               Cls
               Call Lcd_t( "Wahl der BA mit" , "der Taste 2!")
               Ba = 0
            End If

            Waitms 500

            Readeeprom Hz01 , &H03                'bei einer nderung
            If Hz01 <> Ba Then
               Writeeeprom Ba , &H03              'Wert ins EEprom schreiben
            End If

'Taste 3 - lang (1011_xxxx): 176
'-------------------------------
'LCD: Helligkeit der Hintergrundbeleuchtung
         Case 176
            If F_hel = 0 And Pwm_hgb < 20 Then
               Incr Pwm_hgb : Waitms 300
               If Pwm_hgb = 20 Then F_hel = 1     'Richtungsumkehr: HGB dunkler
            Else
               Decr Pwm_hgb : Waitms 300
               If Pwm_hgb = 1 Then F_hel = 0      'Richtungsumkehr: HGB heller
            End If

            Writeeeprom Pwm_hgb , &H02            'Wert im EEprom speichern

'Taste 4 - lang (0111_xxxx): 112
'-------------------------------
'Entsprechend der Betriebsart wird die Funktion aufgerufen.
'  1: Anzeige der Messwerte erfolgt im Hauptprogramm (Spannung, Strom)
'  2: Strom-Abschaltwert verndern ( 50 mA-Schritt)
'  3: Strom-Abschaltwert verndern (250 mA-Schritt)
'  4: Anzeige der Messwerte erfolgt im Hauptprogramm (Widerstand, Leistung)

' 19: Test LCD
' 20: Test Piezo und LED

'In den folgenden Routinen erfolt der Funktionsaufruf ohne weitere Besttigung.
         Case 112
            Cls
            Select Case Ba
'2/3: Strom-Abschaltwert im EEprom speichern
               Case 2 To 3
                  Lcd "Wert im EEprom"
                  Writeeeprom Imax , &H04         'Wert im EEprom speichern

'19: Test LCD
               Case 19
                  Lcd "LCD - Test"
                  Wait 1
                  Call Lcd_t( "X-X-X-X-X-X-X-X-" , "-X-X-X-X-X-X-X-X" )
'20: Test Tongeber und LED
               Case 20
                  Ton = 1
                     Led_rt = 0 : Led_gn = 0      'LED aus
                     Led_gn = 1 : Waitms 500      'LED Farbe grn
                     Led_rt = 1 : Waitms 500      'LED Farbe gelb
                     Led_gn = 0 : Waitms 500      'LED Farbe rot
                     Led_rt = 0 : Waitms 500      'LED aus
                  Ton = 0
                  Lcd "Test beendet"
                Case Else
                  Call Lcd_t( "-" , "(keine Funktion)")
            End Select

            Wait 2
            Ba = 1                                'Betriebsart 1 whlen
      End Select
   End If

'Zeitzhler fr die Lnge der Bettigungszeit zurcksetzen
   If Taste <> Taste_old Then
      Z_taste = 0
   End If
Return

'-------------------------------------------------------------------------------
'40. === I N T E R R U P T R O U T I N E =======================================
'-------------------------------------------------------------------------------

'Timer 0
'-------
Int_tim0:
'T0 = 1/ Quarz* Teiler (1/ 16 MHz* 256* 193 =  1,00 ms; 992 Hz)
'Interrupt: Timer 0 (8 Bit), alle 1 ms
'Voreinstellung des Zhlers
   Timer0 = 193                                   'Voreinstellung (1 ms)
   Incr Z_t0a : Incr Z_t0b                        'Zeitbasis (A, B)

'Um den HLS auch bei geringeren Strmen (ca. < 300 mA) betreiben zu knnen,
'ohne dass das Statussignal "kein Lastwiderstand" zur Abschaltung fhrt,
'wird das Statussignal deaktiviert.
   If Imax < 500 Then
      F_sta = 1                                   'Deaktivierung bei I < 500 mA
   Else
      F_sta = Bts_sta
   End If

   If Z_t0a = 50 Then                             'Zhler A: Kontrast
      Z_t0a = 0
   End If

   If Z_t0b = 20 Then                             'Zhler B: LCD-Beleuchtung
      Z_t0b = 0
   End If

'PWM-Steuerung fr das LCD (Kontrast)
'------------------------------------
'Eine lngere Einschaltzeit bedeutet eine hhere Spannung und damit einen ge-
'ringeren Kontrast des LCD.
   If Z_t0a = 0 Then Lcd_kon = 1                  'LCD-Kontrast .... : Ein
   If Z_t0a = Pwm_kon Then Lcd_kon = 0            'LCD-Kontrast .... : Aus

'PWM-Steuerung fr das LCD (Helligkeit)
'--------------------------------------
'Eine lngere Einschaltzeit bedeutet eine hellere Hintergrundbeleuchtung.
   If F_hgb = 1 Then
      If Z_t0b = 0 Then Lcd_hgb = 1               'LCD-Beleuchtung . : Ein
      If Z_t0b = Pwm_hgb Then Lcd_hgb = 0         'LCD-Beleuchtung . : Aus
   Else
      Lcd_hgb = 0                                 'LCD-Beleuchtung . : Aus
   End If

'Prfung des Stromwertes
'-----------------------
'Abschaltung, wenn der Messwert unterhalb des zulssigen Wertes liegt
         If Hls = 1 And Uimes < Uiab Then         'Grenzwert unterschritten
            Hls = 0                               'HLS ............. : Aus
            F_feh = 1                             'Flag_Strung .... : Ein
         End If
Return
'
'-------------------------------------------------------------------------------
'
'Timer 1
'-------
Int_tim1:
'T1 = 1/ Quarz* Teiler (1/ 16 MHz* 256* 625 = 10,00 ms; 100 Hz)
'Interrupt: Timer 1 (16 Bit), alle 10 ms

'Voreinstellung des Zhlers
   Timer1 = 64911                                 'Voreinstellung (10 ms)

'Tastaturabfrage
   F_taste = 1                                    'Flag setzen

'Zeitzhler
   Incr Z_bl                                      'Zhler: Lebensbit
   Incr Z_t1                                      'Zhler: Zeit (allgemein)
   Incr Z_t2                                      'Zhler: Anzeige

   If Z_t2 = 100 Then                             'Rcksetzen alle s
      Z_t2 = 0
      F_amw = 1                                   'Anzeige der Messwerte
   End If

   If Z_t1 = 10 Then                              'Rcksetzen alle 100 ms
      Z_t1 = 0

'Zeitzhler fr die LCD-Beleuchtung
'----------------------------------
      If Z_hgb > 0 Then
         Decr Z_hgb                               'Zeitzhler: LCD-Beleuchtung
      Else
         F_hgb = 0
      End If

'Zeitzhler fr die LCD-Kontraststeuerung
'----------------------------------------
      If Z_kon > 0 Then Decr Z_kon                'Zeitzhler: LCD-Kontrast
   End If

'Blinken
'-------
      Select Case Z_bl
         Case 0
'Fehler
            If F_feh = 1 Then
               Led_rt = 1                         'LED_rt Ein (Fehler)
               Led_gn = 0
            End If
'HLS oder Statusfehler
            If F_ea = 1 Or F_sta = 0 Then
               If F_feh = 0 Then
                  Led_gn = 1                      'LED_ge Ein (Statusfehler)
                  Led_rt = 1
               End If
            End If
'Lebensbit
            If F_lb = 1 Then
               Led_gn = 1                         'LED_gn Ein (Lebensbit)
            End If
         Case 1
'LED ausschalten bei HSL aus
            If F_ea = 0 And F_feh = 0 And F_sta = 1 Then
               Led_gn = 0
               Led_rt = 0
            End If
'LED ausschalten
         Case 10
            If F_lb = 1 Then
               Led_gn = 0                         'LED_gn Aus (Lebensbit)
            End If
         Case 200
            If F_feh = 1 Then
               Led_rt = 0                         'LED_rt Aus (Fehler)
            End If
            If F_sta = 0 Then                     'LED_ge Aus (Statusfehler)
               Led_gn = 0
               Led_rt = 0
            End If
         Case Else
            If F_feh = 1 Then
               Led_gn = 0
            End If
      End Select
Return

'-------------------------------------------------------------------------------
'50. === U N T E R P R O G R A M M - C A L L ===================================
'-------------------------------------------------------------------------------

'Hintergrundbeleuchtung
'----------------------
'LCD-Modul: Hintergrundbeleuchtung einschalten; bergabe: Zeitdauer [ms]
Sub Lcd_b(byval T_hgb As Byte )
'Zeit-Zhler initialisieren
  If T_hgb > Z_hgb Then
     Z_hgb = T_hgb                                'LCD-Beleuchtungszeit
  End If

  F_hgb = 1
End Sub

'Tonausgabe
'----------
'Ton fr eine Zeitdauer einschalten; bergabe: Zeitdauer [ms]
Sub Ton_d(byval T_ton As Word )
   Ton = 1
   Waitms T_ton
   Ton = 0
End Sub

'LCD-Werte normieren
'-------------------
Sub Wanz(byval Wsin As Single)
   Hs01 = Fusing(wsin , "#.##")                   'formatierte Ausgabe
M_w01:
   If Len(hs01) < 6 Then
      Hs01 = " " + Hs01
      Goto M_w01
   End If
End Sub

'LCD-Anzeige
'-----------
Sub Lcd_t(byval Tul As String , Byval Tll As String)
'Die zwei Zeilen des LCD werden unterschieden
'1. Zeile
   Tul_1 = Tul
   If Tul_1 = "-" Then Goto M1_lcd                'keine Ausgabe bei "-"
Lcd_t1:
      If Len(tul_1) < 16 Then                     'normierte Lnge
         Tul_1 = Tul_1 + " "
         Goto Lcd_t1
      End If

      If Tul_1 <> Tul_old Then
         Tul_old = Tul_1
         Locate 1 , 1                             'Zeile 1, Position 1
         Lcd Tul_1
      End If

'2. Zeile
M1_lcd:
   Tll_1 = Tll
   If Tll_1 = "-" Then Goto M2_lcd                'keine Ausgabe bei "-"
Lcd_t2:
      If Len(tll_1) < 16 Then                     'normierte Lnge
         Tll_1 = Tll_1 + " "
         Goto Lcd_t2
      End If

      If Tll_1 <> Tll_old Then
         Tll_old = Tll_1
         Locate 2 , 1                             'Zeile 2, Position 1
         Lcd Tll_1
      End If
M2_lcd:
End Sub

'-------------------------------------------------------------------------------
'60. === F U N K T I O N =======================================================
'-------------------------------------------------------------------------------

'-------------------------------------------------------------------------------
'70. === D A T E N B E R E I C H ===============================================
'-------------------------------------------------------------------------------

'Ende

'###############################################################################
